--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Characters
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Character class definition
LCharacter =
{
	__tostring = function(t) return "{Character:" .. t.name .. "}" end,
	_ByName = {},
	
--	action = { Speak { "speak_default" } },		-- default behavior
	action = {},
}

-- Mood constants
LCharacter.kHappy = 70			-- "Happy" threshold
LCharacter.kAngry = 25			-- "Angry" threshold
LCharacter.kHaggleDelta = 4		-- Amount of mood change when Haggling
LCharacter.kPurchaseDelta = 10	-- Amount of mood change when making a purchase
LCharacter.kDefault = 50

-------------------------------------------------------------------------------
-- Functions for data description

function DefineCharacter(t) return LCharacter:new(t) end

-------------------------------------------------------------------------------
-- "static" functions to access global lists

function LCharacter:ByName(name)
	return self._ByName[name]
end

-------------------------------------------------------------------------------
-- "constructor"

function LCharacter:new(t)
	-- Standard object creation...
	t = t or {} setmetatable(t, self) self.__index = self
	
	-- Check parameters
	if not t.name then
		bsgDevWarning("Character defined with no name")
	elseif self._ByName[t.name] then
		bsgDevWarning("Character " .. t.name .. " already defined")
	else
		-- Keep global tables
		self._ByName[t.name] = t
		
		-- Default emotional state
		t.defmood = t.defmood or LCharacter.kDefault
		t.mood = t.defmood
		t.moodWeek = 0
		
		-- Actions - make sure it's a table of functions
		if type(t.action) == "function" then t.action = { t.action } end
		for _,a in ipairs(t.action) do
			if type(a) ~= "function" then
				bsgDevWarning("Character " .. t.name .. " contains a bad action: " .. tostring(a))
				return nil
			end
		end
		
		-- List of quests starting/ending with this character
		t.quests = {}
		t.endQuests = {}
		
		return t
	end
	
	return nil
end

-------------------------------------------------------------------------------
-- Graphics and Popups

function LCharacter:Visual(x,y,font)
--	local yText = 193
	local yText = 231 - 11
	font = font or charNameFontDark
	local h = bsgFontHeight(font) * 2
--	return Rollover { x=x,y=y, w=ui.charW,h=ui.charH, fit=true,
--		contents=self:RolloverTarget(),
	return Window { x=x,y=y, w=ui.charW,h=ui.charH, fit=true,
		Character { x=8,y=0,w=ui.charW,h=ui.charH, char=self },
		Bitmap { x=0,y=yText, image="image/name_tag",
--			Text { x=0,y=yText,w=ui.charW,h=h, font=font,
			Text { x=17,y=15,w=156,h=19, font=font,
				label=LabelString(self.name), flags=kVAlignCenter+kHAlignCenter }, },
	}
end

function LCharacter:RolloverTarget()
	return "(LCharacter:ByName('"..self.name.."')):Rollover()"
end

function LCharacter:Rollover()
	return MakeDialog
	{
		Rectangle
		{
			x=0,y=0, color=PopupColor, inset=2,
			AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
			TightText { x=0,y=0, label=self.name },

	-- TODO: Emotional state or other notes
--			TightText { x=0,y=bsgFontHeight(popupFont), label="#DEBUG-MOOD:"..tostring(self.mood) }
		},
	}
end

-------------------------------------------------------------------------------
-- Character mood basics

-- Every week, characters tend toward "neutral"
function LCharacter:MoodTick()
	for _,char in pairs(self._ByName) do
		if char.mood > char.defmood then char:SetMood(char.mood - 1)
		elseif char.mood < char.defmood then char:SetMood(char.mood + 1)
		end
	end
end

function LCharacter:IsHappy()
	return (self.mood >= LCharacter.kHappy)
end

function LCharacter:IsNeutral()
	return (LCharacter.kAngry < self.mood and self.mood < LCharacter.kHappy)
end

function LCharacter:IsAngry()
	return (self.mood <= LCharacter.kAngry)
end

-------------------------------------------------------------------------------
-- Quest ending

function LCharacter:EndsQuest(q)
	for _,endQ in ipairs(self.endQuests) do
		if q == endQ then return true end
	end
	return false
end

-------------------------------------------------------------------------------
-- Reset

function LCharacter:ResetAll()
	for name,c in pairs(LCharacter._ByName) do
		-- Set default emotional state
		c.mood = c.defmood or LCharacter.kDefault
		c.moodWeek = 0
	end
end

-------------------------------------------------------------------------------
-- Load and Save

-- TODO: Load/Save character moodWeek or allow this loophole??
-- (make a transaction to raise character mood, quit, rester, make another transation to raise character mood)

function LCharacter:BuildSaveTable()
	local t = {}
	for _,c in pairs(self._ByName) do
		if c.mood ~= c.defmood then
			t[c.name] = c.mood
		end
	end
	return t
end

function LCharacter:LoadSaveTable(t)
	for name,value in pairs(t) do
		local c = LCharacter:ByName(name)
		if c then c.mood = value end
	end
end

-------------------------------------------------------------------------------
-- Mood

function LCharacter:SetMood(mood)
	if mood < 0 then mood = 0
	elseif mood > 100 then mood = 100
	end
	
	self.mood = mood
	SetCharacterMood(self.name, mood)
end

-------------------------------------------------------------------------------
-- Actions

function LCharacter:DoAction()
	-- Pick an action at random and do it
	local i = table.getn(self.action)
	i = bsgRandom(1,i)
	local f = self.action[i]
	local done = false
	if f then done = f(self) end
	
	-- If the action didn't succeed, try every action...
	if not done then
		for i=1,table.getn(self.action) do
			f = self.action[i]
			done = f(self)
			if done then break end
		end
	end
	
	-- Worst case, do Speak{"speak_default"} equivalent
	if not done then
		if gSim.quest then
			local k = GetString("speak_default_quest")
			DisplayDialog { "ui/chartalk.lua", char=self, body="#"..k, ok="ok" }
		else
			local k = GetString("speak_default")
			DisplayDialog { "ui/chartalk.lua", char=self, body="#"..k, ok="ok" }
		end
	end
end
